<?php
include 'db_connect.php';

header("Content-Type: application/json; charset=UTF-8");

$input = json_decode(file_get_contents('php://input'), true);

$username = $input['username'] ?? '';
$password = $input['password'] ?? '';
$fullName = $input['fullName'] ?? '';
$email = $input['email'] ?? '';

// --- Validación Básica ---
if (empty($username) || empty($password) || empty($fullName) || empty($email)) {
    echo json_encode(['status' => 'error', 'message' => 'Todos los campos son obligatorios.']);
    exit();
}
if (strlen($password) < 8) {
    echo json_encode(['status' => 'error', 'message' => 'La contraseña debe tener al menos 8 caracteres.']);
    exit();
}
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['status' => 'error', 'message' => 'El formato del correo electrónico no es válido.']);
    exit();
}

// --- Comprobar si el usuario o el correo ya existen ---
$check_sql = "SELECT username, email FROM users WHERE username = ? OR email = ?";
$stmt = $conn->prepare($check_sql);
$stmt->bind_param("ss", $username, $email);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows > 0) {
    $existingUser = $result->fetch_assoc();
    if ($existingUser['username'] === $username) {
        echo json_encode(['status' => 'error', 'message' => 'El nombre de usuario ya está en uso.']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'El correo electrónico ya está registrado.']);
    }
    $stmt->close();
    $conn->close();
    exit();
}
$stmt->close();

// --- Hashear la contraseña e insertar el nuevo usuario ---
$hashed_password = password_hash($password, PASSWORD_DEFAULT);

$insert_sql = "INSERT INTO users (username, password, full_name, email) VALUES (?, ?, ?, ?)";
$stmt = $conn->prepare($insert_sql);
$stmt->bind_param("ssss", $username, $hashed_password, $fullName, $email);

if ($stmt->execute()) {
    $new_user_id = $conn->insert_id;
    $stmt->close();

    // Empezamos una transacción para asegurar que todo (matrículas y notas) se haga correctamente.
    $conn->begin_transaction();

    try {
        // Obtenemos los IDs de todos los cursos
        $courses_result = $conn->query("SELECT id FROM courses");
        $course_ids = [];
        while ($row = $courses_result->fetch_assoc()) {
            $course_ids[] = $row['id'];
        }

        // Definimos las notas por defecto que se crearán para cada curso
        $default_grades = [
            "Práctica Calificada 1" => "0.00",
            "Práctica Calificada 2" => "0.00",
            "Examen Parcial" => "0.00",
            "Examen Final" => "0.00"
        ];

        // Preparamos las consultas que se usarán repetidamente
        $enroll_stmt = $conn->prepare("INSERT INTO enrollments (user_id, course_id) VALUES (?, ?)");
        $grade_stmt = $conn->prepare("INSERT INTO grades (user_id, enrollment_id, course_id, grade_name, grade_value) VALUES (?, ?, ?, ?, ?)");

        // Para cada curso, matriculamos al usuario y creamos sus notas
        foreach ($course_ids as $course_id) {
            // 1. Matricular
            $enroll_stmt->bind_param("ii", $new_user_id, $course_id);
            $enroll_stmt->execute();
            $new_enrollment_id = $conn->insert_id; // Obtenemos el ID de esta matrícula

            // 2. Crear notas por defecto para esta matrícula
            foreach ($default_grades as $grade_name => $grade_value) {
                $grade_stmt->bind_param("iiiss", $new_user_id, $new_enrollment_id, $course_id, $grade_name, $grade_value);
                $grade_stmt->execute();
            }
        }

        // Si todo salió bien, confirmamos los cambios
        $conn->commit();
        echo json_encode(['status' => 'success', 'message' => '¡Registro exitoso! Ya estás matriculado y con notas inicializadas.']);

    } catch (Exception $e) {
        // Si algo falló, revertimos todos los cambios de la transacción
        $conn->rollback();
        error_log("PHP_ERROR en register_user: Transacción fallida: " . $e->getMessage());
        echo json_encode(['status' => 'success', 'message' => 'Usuario registrado, pero falló la matriculación automática.']);
    }

    // Cerramos las consultas preparadas
    $enroll_stmt->close();
    $grade_stmt->close();

} else {
    // Error al insertar el usuario
    error_log("PHP_ERROR en register_user: Falló la inserción de usuario: " . $stmt->error);
    echo json_encode(['status' => 'error', 'message' => 'Error al registrar el usuario en la base de datos.']);
    $stmt->close();
}

$conn->close();

?>